/////////////////////////////////////////////////////////////////////////////////

// Original obtained from ShaderToy.com
// Adapted, trivialy, for VGHD by TheEmu.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed
#define iResolution u_WindowSize
#define iMouse AUTO_MOUSE

/////////////////////////////////////////////////////////////////////////////////

// Simple "Automatic Mouse". Simulates scanning the mouse over the full range of
// the screen with the X and Y scanning frequencies being different. TheEmu.

#define MOUSE_SPEED vec2(0.2,0.0)
#define MOUSE_POS   vec2((1.0+sin(iGlobalTime*MOUSE_SPEED))*u_WindowSize/2.0)
#define MOUSE_PRESS vec2(0.0,0.0)
#define AUTO_MOUSE  vec4( MOUSE_POS, MOUSE_PRESS )

/////////////////////////////////////////////////////////////////////////////////

// The ShaderToy shaders often use textures as inputs named iChannel0. With VGHD
// this may access a Sprite, ClipSprite or ClipNameSprite image depending on how
// the .scn file declares them.
//
// Note, the name used here does not seem to make any difference, so I have used
// iChannel0 which is what is used by ShaderToy but you can use any name as long
// as it matches the use in the main body of the shader. TheEmu.

uniform sampler2D iChannel0;

// With VGHD the range of the P argument's components of the texture functions is
// 0.0 to 1.0 whereas with ShaderToy it seems that the upper limits are given  by
// the number of pixels in each direction, typically 512 or 64.  We therefore use
// the following functions instead.

vec4 texture2D_Fract(sampler2D sampler,vec2 P) {return texture2D(sampler,fract(P));}
vec4 texture2D_Fract(sampler2D sampler,vec2 P, float Bias) {return texture2D(sampler,fract(P),Bias);}

// Rather than edit the body of the original shader we use use a define  here  to
// redirect texture calls to the above functions.

#define texture2D texture2D_Fract

/////////////////////////////////////////////////////////////////////////////////

#ifdef GL_ES
precision highp float;
#endif

// Rendering parameters
#define RAY_LENGTH_MAX		100.0
#define RAY_BOUNCE_MAX		20
#define RAY_STEP_MAX		40
#define COLOR				vec3 (0.8, 0.8, 0.9)
#define LIGHT				vec3 (1.0, 1.0, -1.0)
#define REFRACT_FACTOR		0.8
#define REFRACT_INDEX		2.417
#define AMBIENT				0.2
#define SPECULAR_POWER		3.0
#define SPECULAR_INTENSITY	0.5

// Math constants
#define DELTA	0.001
#define PI		3.14159265359

// Rotation matrix
mat3 mRotate (in vec3 angle) {
	float c = cos (angle.x);
	float s = sin (angle.x);
	mat3 rx = mat3 (1.0, 0.0, 0.0, 0.0, c, s, 0.0, -s, c);

	c = cos (angle.y);
	s = sin (angle.y);
	mat3 ry = mat3 (c, 0.0, -s, 0.0, 1.0, 0.0, s, 0.0, c);

	c = cos (angle.z);
	s = sin (angle.z);
	mat3 rz = mat3 (c, s, 0.0, -s, c, 0.0, 0.0, 0.0, 1.0);

	return rz * ry * rx;
}

// Rotation matrix (rotation on the Y axis)
vec3 vRotateY (in vec3 p, in float angle) {
	float c = cos (angle);
	float s = sin (angle);
	return vec3 (c * p.x - s * p.z, p.y, c * p.z + s * p.x);
}

// Distance to the scene
vec3 normalTopA = normalize (vec3 (0.0, 1.0, 1.4));
vec3 normalTopB = normalize (vec3 (0.0, 1.0, 1.0));
vec3 normalTopC = normalize (vec3 (0.0, 1.0, 0.8));
vec3 normalBottomA = normalize (vec3 (0.0, -1.0, 1.4));
vec3 normalBottomB = normalize (vec3 (0.0, -1.0, 1.8));
float getDistance (in vec3 p) {
	p = mRotate (vec3 (iGlobalTime)) * p;

	float topCut = p.y - 1.1;
	float angleStep = PI / (2.0 + floor (16.0 * iMouse.x / iResolution.x));
	float angle = angleStep * (0.5 + floor (atan (p.x, p.z) / angleStep));
	vec3 q = vRotateY (p, angle);
	float topA = dot (q, normalTopA) - 2.0;
	float bottomA = dot (q, normalBottomA) - 2.0;
	float topC = dot (q, normalTopC) - 1.8;
	q = vRotateY (p, -angleStep * 0.5);
	angle = angleStep * floor (atan (q.x, q.z) / angleStep);
	q = vRotateY (p, angle);
	float bottomB = dot (q, normalBottomB) - 1.95;
	float topB = dot (q, normalTopB) - 1.92;

	return max (topCut, max (topA, max (topB, max (topC, max (bottomA, bottomB)))));
}

// Normal at a given point
vec3 normal (in vec3 p) {
	const vec2 h = vec2 (DELTA, -DELTA);
	return normalize (
		h.xxx * getDistance (p + h.xxx) +
		h.xyy * getDistance (p + h.xyy) +
		h.yxy * getDistance (p + h.yxy) +
		h.yyx * getDistance (p + h.yyx)
	);
}

// Cast a ray for a given color channel (and its corresponding refraction index)
vec3 lightDirection = normalize (LIGHT);
float raycast (in vec3 origin, in vec3 direction, in float refractIndex, in vec3 channel) {
	float color = 0.0;
	float intensity = 1.0;
	float distanceFactor = 1.0;
	float refractionRatio = 1.0 / refractIndex;

	float rayLength = 0.0;
	for (int rayBounce = 0; rayBounce < RAY_BOUNCE_MAX; ++rayBounce) {

		// Ray marching
		float dist = RAY_LENGTH_MAX;
		for (int rayStep = 0; rayStep < RAY_STEP_MAX; ++rayStep) {
			dist = distanceFactor * getDistance (origin);
			float distMin = max (dist, DELTA);
			rayLength += distMin;
			if (dist < 0.0 || rayLength > RAY_LENGTH_MAX) {
				break;
			}
			origin += direction * distMin;
		}

		// Check whether we hit something
		if (dist >= 0.0) {
			break;
		}

		// Get the normal
		vec3 normal = distanceFactor * normal (origin);

		// Basic lighting
		vec3 reflection = reflect (direction, normal);
		if (distanceFactor > 0.0) {
			float relfectionDiffuse = max (0.0, dot (normal, lightDirection));
			float relfectionSpecular = pow (max (0.0, dot (reflection, lightDirection)), SPECULAR_POWER) * SPECULAR_INTENSITY;
			float localColor = (AMBIENT + relfectionDiffuse) * dot (COLOR, channel) + relfectionSpecular;
			color = mix (color, localColor, intensity);
			intensity *= REFRACT_FACTOR;
		}

		// Next ray...
		vec3 refraction = refract (direction, normal, refractionRatio);
		if (dot (refraction, refraction) < DELTA) {
			direction = reflection;
			origin += direction * DELTA * 2.0;
		} else {
			direction = refraction;
			distanceFactor = -distanceFactor;
			refractionRatio = 1.0 / refractionRatio;
		}
	}

	// Get the background color
	// float backColor = dot (textureCube (iChannel0, direction).rgb, channel);
	// Return the intensity of this color channel
	// Return mix (color, backColor, intensity);

   // Modified version, because cube textures not supprted by VGHD. Also we
   // suppress the actual background to have a standalone diamond. TheEmu.

   float backColor = dot ( texture2D(iChannel0,direction.xy).rgb, channel );
	float result = mix (color, backColor, intensity);

   return ( result == backColor ) ? 0.0 : result;
}

// Main function
void main ( void ) {

	// Define the ray corresponding to this fragment
	vec2 frag = (2.0 * gl_FragCoord.xy - iResolution.xy) / iResolution.y;
	vec3 direction = normalize (vec3 (frag, 2.0));

	// Set the camera
	vec3 origin = 8.0 * vec3 ((cos (iGlobalTime * 0.1)), sin (iGlobalTime * 0.2), sin (iGlobalTime * 0.1));
	vec3 forward = -origin;
	vec3 up = vec3 (sin (iGlobalTime * 0.3), 2.0, 0.0);
	mat3 rotation;
	rotation [2] = normalize (forward);
	rotation [0] = normalize (cross (up, forward));
	rotation [1] = cross (rotation [2], rotation [0]);
	direction = rotation * direction;

	// Cast a ray for each color channel
	gl_FragColor.r = raycast (origin, direction, REFRACT_INDEX + 0.00, vec3 (1.0, 0.0, 0.0));
	gl_FragColor.g = raycast (origin, direction, REFRACT_INDEX + 0.05, vec3 (0.0, 1.0, 0.0));
	gl_FragColor.b = raycast (origin, direction, REFRACT_INDEX - 0.05, vec3 (0.0, 0.0, 1.0));
	gl_FragColor.a = length(gl_FragColor.rgb);
}